!=====================================================================!
!                                                                     !
!                                                                     !
!                        T O P O    (gfortran)                        !
!                                                                     !
!                                                                     !
!   Programm zur Berechnung des Erdvolumens und weiterer Parameter    !
!   unter Beruecksichtigung der Land- und Wassermassen ueber Meeres-  !
!   spiegelniveau bei Verwendung der topographischen Daten von        !
!   "Worldbath" und unter Beruecksichtigung der Ellipsoid- bzw.       !
!   Sphaeroidgestalt der Erde.                                        !
!                                                                     !
!                            Hans Jelitto                             !
!                       Hamburg, 3. August 2025                       !
!                                                                     !
!                                                                     !
!   Zum Programm gehoert die Datei der topographischen Hoehendaten    !
!   '[X]data.tsv'. Sie wurde von der Website (Worldbath):             !
!                                                                     !
!      http://iridl.ldeo.columbia.edu/SOURCES/.WORLDBATH/.bath/       !
!                                                                     !
!   unter 'Data Files' und 'Text with tab-separated-values' [X+]      !
!   heruntergeladen.                                                  !
!                                                                     !
!   Als simplen Ansatz koennte man annehmen, dass die durchschnitt-   !
!   liche Hoehe zur Berechnung des Volumens der Landmassen genuegt.   !
!   Leider ist dieser Wert fuer die Berechnung des Erdvolumens nicht  !
!   brauchbar, da nur der Mittelwert der Daten berechnet wird. Dies   !
!   ist fuer kleine Gebiete, in denen sich der Abstand der geogra-    !
!   phischen Gitterpunkte kaum aendert, zulaessig. Ueber groessere    !
!   Gebiete und speziell inklusive der Pole liefert diese Methode     !
!   jedoch keine korrekten Ergebnisse, da die unterschiedlichen       !
!   Gitterpunktabstaende nicht beruecksichtigt werden. Zum Vergleich  !
!   wird dennoch dieser vereinfachte Wert durch das Programm mit      !
!   berechnet und als arithmetisches Mittel in Klammern angegeben.    !
!   Entsprechende Zahlen fuer die Landmassen ueber dem Meeresspiegel  !
!   sind unter "(arithm. mean)" aufgefuehrt. Sie sind jedoch auch     !
!   meist nicht korrekt wegen dieses zu einfachen Rechenansatzes.     !
!                                                                     !
!   Darueber hinaus kann die normale Mittelwertbestimmung sowieso     !
!   nicht verwendet werden, weil zur Berechnung des Erdvolumens in    !
!   unserem Fall auf den Meeresflaechen nicht der Meeresboden, son-   !
!   dern der Meeresspiegel als Grundlage dient.                       !
!                                                                     !
!   Die Zahlen unter "vol.-based" entsprechen der mittleren Hoehe     !
!   gemaess dem tatsaechlichen Volumen. Die Berechnung wurde abge-    !
!   leitet aus Gleichung (78) mit gemitteltem Erdradius und Vernach-  !
!   laessigung des Terms hoechster Ordnung (h^3). Darueber hinaus     !
!   existieren zwei alternative Berechnungsmethoden (in Kommentare    !
!   umgewandelt - siehe unten "!c"). Die Zahlen unter "ell.-based"    !
!   (ellipsoid-based) beruhen auf einem einfacheren Ansatz "Flaeche   !
!   mal Hoehe", das heisst ohne Volumeneffekte hoeherer Ordnung.      !
!                                                                     !
!   In diesem Programm werden die groessten Binnenseen mit Ober-      !
!   flaechen, welche sich ueber- oder unterhalb des Meeresspiegels    !
!   befinden, ebenfalls erfasst. Bei einigen Seen in Worldbath ist    !
!   der Boden und bei anderen Seen die Wasserspiegelhoehe angegeben.  !
!   Insgesamt betraegt die Volumenkorrektur, die sich durch solche    !
!   Seen mit Angabe Seeboden ergibt, ueber 2600 Kubikkilometer, was   !
!   prozentual gesehen jedoch immer noch gering ist.                  !
!                                                                     !
!                        ---------------------                        !
!                                                                     !
!        Zum Programm 'Topo' gehoeren nachfolgende 6 Dateien:         !
!                                                                     !
!        Datei                  Kurzbeschreibung                      !
!   --------------------------------------------------------------    !
!      topo          Ausfuehrbare Programmdatei                       !
!      topo.f95      FORTRAN-Quellcode, vorliegender Text             !
!      [X]data.tsv   Topograph. Daten (5 minute-grid, Worldbath)      !
!      zlakes.txt    Korrekturdaten, hoch und tief liegende Seen      !
!      readme.pdf    Kurzinformation zum Programm                     !
!      out.txt       Ergebnis-Datei (wird mit jedem Programmlauf      !
!                    ueberschrieben)                                  !
!   --------------------------------------------------------------    !
!                                                                     !
!=====================================================================!

      module constants; real(8) :: pi,a,c,dl,dnx
        integer(4), parameter :: NX5 = 4320, NY5 = 2161
      end module
      module lakes; real(8) :: se(5,50),zero
        integer(4) :: il(5,50),nlake
      end module

      program topo
!-----Hauptprogramm----------------------------------------------------

      use constants; use lakes
      implicit double precision (A-H,O-Z); integer(4) ivap(NX5)
!     Bei folgenden Deklarationen (0:2) bedeuten die Indizes drei
!     verschiedene Hoehen: "0" bedeutet Meeresboden, "1" Meeresspiegel
!     und "2" Meeresspiegel mit hoch und tief liegenden Binnenseen.
      real(8) ev(0:2),hmitt(0:2),hmitv(0:2),hmitw(0:2)
      real(8) sumh(0:2),sumo(0:2),sumw(0:2),Vg(0:2),Vges(0:2)
      character(38) :: title1,title2,line
      character(23) :: dummy; character(11) :: tfile

!-----Parameter eingeben
      title1 = '   EARTH`S VOLUME INCLUDING LANDMASS  '
      title2 = ' (Worldbath, 5 arc-minute resolution) '
      line   = '======================================'
      open(unit=1,file='out.txt')
      write(6,'(//4(21x,A38/))') line,title1,title2,line
      write(1,'(/2(21x,A38/))') title1,title2
      write(*,'(17x,''latitude .......... (min. -90.0) from :  '')' &
     &      ,advance='no')
      read(*,*) gbmin
      if (gbmin<-90.d0 .or.gbmin>90.d0) go to 100
      write(*,'(17x,''latitude .......... (max.  90.0)   to :  '')' &
     &      ,advance='no')
      read(*,*) gbmax
      if(gbmax<-90.d0 .or.gbmax>90.d0 .or.gbmax<gbmin) go to 100
      write(*,'(17x,''ext. longitude east (min.   0.0) from :  '')' &
     &      ,advance='no')
      read(*,*) glmin
      if (glmin<0.d0 .or.glmin>360.d0) go to 100
      write(*,'(17x,''ext. longitude east (max. 360.0)   to :  '')' &
     &      ,advance='no')
      read(*,*) glmax
      if (glmax<0.d0 .or.glmax>360.d0 .or.glmax<glmin) go to 100

!-----5x5-Bogenminuten-Gitter (Worldbath)
      tfile = '[X]data.tsv'
      NX = NX5; NY = NY5

!-----Programmstart
      dnx   = dfloat(NX)
      dgrad = dnx/360.d0
      ibmin = idnint((gbmin + 90.d0)*dgrad + 1.d0)
      ibmax = idnint((gbmax + 90.d0)*dgrad + 1.d0)
      ilmin = idnint(glmin*dgrad + 1.d0)
      ilmax = idnint(glmax*dgrad)
      if (ilmax<ilmin) go to 100
      db    = dfloat(ibmax-ibmin+1)
      dl    = dfloat(ilmax-ilmin+1)
      write(*,'(/''   Computation started.  '')',advance='no')
      do i=1,idint((gbmax+90.d0)/5.d0)
        write(*,'(''>'')',advance='no')
      enddo
      write(*,'(/''   Output file: out.txt  '')',advance='no')

!-----Einlesen der Daten hoch bzw. tief liegender Seen
      zero = 0.d0
      open(unit=10,file='zlakes.txt')
      do i=1,14
        read(10,*)
      enddo
      read(10,'(A23,I4)') dummy,nlake
      do i=1,5
        read(10,*)
      enddo
      do i=1,nlake
        read(10,*)
        read(10,*) (se(j,i),j=1,5)
      enddo
      do i=1,nlake
        do j=1,2
          il(j,i) = idnint((se(j,i)+ 90.d0)*dgrad + 1.d0)
        enddo
        if (se(3,i)>=zero) then
          il(3,i) = idnint(se(3,i)*dgrad + 1.d0)
        else
          il(3,i) = idnint((se(3,i)+360.d0)*dgrad + 1.d0)
        endif
        if (se(4,i)>=zero) then
          il(4,i) = idnint(se(4,i)*dgrad)
        else
          il(4,i) = idnint((se(4,i)+360.d0)*dgrad)
        endif
!c      write(*,*)'i, il(1..5) = ',i,(il(j,i),j=1,4),se(5,i)
      enddo
      close(10)

!-----Weitere Konstanten
!                                          Aequatorradius  Polradius
!     Referenz-Ellipsoid Erde:                  a [m]       c [m]
!     ================================================================
!     K.R. Lang, Astophys. Data, Planets ...   6378140     6356775
!     World Geod. System's Ref. Ellipsoid      6378137     6356752.3
!     IERS Conventions (1989)                  6378136     6356751.3
!     IERS Conventions (2003)                  6378136.6   6356751.9
!     ================================================================
      a     = 6378136.6d0
      c     = 6356751.9d0
      pi    = 3.14159265358979324d0
      V0    = (4.d0*pi/3.d0)*a**2*c * 1.d-9
! . . Ellipsoid-Oberflaeche, Hauptachsen: a = b > c (oblat)
      e     = dsqrt(1.d0 - (c/a)**2)
      ath   = 0.5d0 * dlog((1.d0 + e)/(1.d0 - e))
      Feli  = 2.d0 * pi * (a*a + c*c * ath/e)  ! [m**2]
! . . Flaeche an den Polen (fuer -90 und 90 Grad geogr. Breite)
      dwi   = pi/(180.d0*dgrad)
      call koord(-0.5d0*pi+0.5d0*dwi,x0,y0)
      Fpol  = pi * x0**2
! . . Kugel-Oberflaeche bei gleichem Volumen wie Ellipsoid
      rm    = (a**2*c)**(1.d0/3.d0)
      Fkug  = 4.d0*pi*rm**2  ! [m**2]
! . . Weitere Initialisierungen
      imo   = 5 * idnint(dgrad)
      inum  = 0
      Fges  = zero
      do i = 0,2
        sumw(i) = zero
        Vg(i)   = zero
        Vges(i) = zero
      enddo

!-----Berechnung - Volumen der Landmassen und Erdoberflaeche
      open(unit=5,file=tfile,status='unknown', &
     &     access='sequential',RECL=NX*NY)
      do k=1,ibmax
        if (mod(k,imo).eq.0) write(*,'(''>'')',advance='no')
        read(5,*) (ivap(i),i=1,ilmax)
        if (k>=ibmin) then
          do i=0,2; sumh(i) = zero; sumo(i) = zero; enddo
          breite = -0.5d0*pi + dfloat(k-1)*dwi
          call koord(breite,x,y)
          r = dsqrt(x**2 + y**2)
! . . . . Innere Hauptschleife (Beruecksichtigung des Meeresspiegels)
          do l=ilmin,ilmax
            ev(0) = dfloat(ivap(l))
            ev(1) = ev(0)
            if (ev(1)<zero) ev(1) = zero
            call levels(ev(0),k,l,ev(2))
            do i=0,2
              ek = ev(i)
              sumh(i) = sumh(i) + ek
! . . . . . . mit Volumenanteilen hoeherer Ordnung
              sumo(i) = sumo(i) + ek*(1.d0 + ek/r + ((ek/r)**2)/3.d0)
            enddo
            inum = inum + 1
          enddo
! . . . . Flaechenberechnung (Meeresspiegel)
          call stripe(breite,dwi,100,Fi,Fh)
          if (k.eq.ibmin) Fi = Fi - Fh
          if (k.eq.ibmax) Fi = Fh
          if (k.eq.1.or.k.eq.NY) Fi = Fpol * dl/dnx
          Fges = Fges + Fi
          do i=0,2
            sumw(i) = sumw(i) + sumh(i)/dl
! . . . . . Vg: Volumen zur Berechnung der mittleren Hoehe
            Vg(i)   = Vg(i)   + Fi*sumh(i)/dl
            Vges(i) = Vges(i) + Fi*sumo(i)/dl
          enddo
        endif
      enddo
      close(5)

!-----Ergebnisse der mittleren Hoehe
      do i=0,2
! . . . Mittlere Hoehe nach Flaechenanteilen auf Meeresspiegelniveau
!       (Naeherung gemaess Volumen = Flaeche * Hoehe)
        hmitt(i) = Vg(i)/Fges

! - - - 1. Mittlere Hoehe gemaess Volumen: Berechnung abgeleitet aus
!       Gleichung (78) mit mittlerem Erdradius und Vernachlaessigung
!       des Terms hoechster Ordnung (Hoehe^3)
        hmitv(i) = -rm/2.d0 + dsqrt(Vges(i)*rm/Fges + rm**2/4.d0)
!       write(*,'(/3x,a7,f10.3)')'1.  h =',hmitv(i)

! . . . 2. Alternative Berechnung mit Ellipsoidform und auch Vernach-
!       laessigung des Terms hoechster Ordnung (Hoehe^3)
!c      cube = 0.75d0*Vges(i)/pi
!c      q = a*(a/2.d0 + c)/(2.d0*a + c)
!c      hmitv(i) = (-q + dsqrt(cube/(2.d0*a + c) + q**2)) * Feli/Fges
!       write(*,'(3x,a7,f10.3)')'2.  h =',hmitv(i)

! . . . 3. Alternative Berechnung mit Kugelvolumen und Korrektur ge-
!       maess Ellipsoid-Oberflaeche (bei sehr kleiner Flaeche ungenau
!       bzw. falsch, wie z.B. bei einem oder wenigen Punkten nahe Pol)
!c      cube = rm**3 + 0.75d0*Vges(i)/pi
!c      hmitv(i) = (-rm + cube**(1.d0/3.d0)) * Fkug/Fges
!       write(*,'(3x,a7,f10.3)')'3.  h =',hmitv(i)

! . . . Arithmetisches Mittel
        hmitw(i) = sumw(i)/db
        Vges(i) = Vges(i)*1.d-9
      enddo
! . . Ergebnisse (Flaeche und Volumen)
      Fges  = Fges*1.d-6        ! Gesamtflaeche (aufintegriert)
      Verdl = Vges(2) - Vges(1) ! Volumen der hoeher gelegenen Seen
      Verdg = V0 + Vges(1)      ! Erdvolumen (Ellipsoid) + Landmassen
      Verds = V0 + Vges(2)      ! Erdvolumen + Landmassen + h. g. Seen

!-----Ergebnis-Ausgabe
      write(*,'(/)')
      do iu=1,6,5
       write(iu,'(1x,78a1)') ('=',i=1,78)
       write(iu,'(7x,A26,f13.1,A29)')'Earth`s equator. radius  :', &
     &  a,' m   (IERS conventions, 2003)'
       write(iu,'(7x,A26,f13.1,A29)')'Earth`s polar radius     :', &
     &  c,' m   (IERS conventions, 2003)'
       write(iu,'(7x,A26,f13.1,A2)')'Earth`s mean radius      :', &
     &  rm,' m'
       write(iu,'(7x,A13,A16,f8.2,A4,f8.2,3X,A16)')'Geograph. lat', &
     &  'itude [deg] :   ',gbmin,'  to',gbmax,'used grid points'
       write(iu,'(7x,A13,A16,f8.2,A4,f8.2,5X,I14/)')'Extended long', &
     &  'itude [deg] :   ',glmin,'  to',glmax,inum
       write(iu,'(A33,A44)') '       Kind of average           ', &
     &  '  ell.-based    vol.-based   (arithm. mean)'
       write(iu,'(7x,70a1)') ('-',i=1,70)
       write(iu,'(A33,F11.3,A2,F12.3,A6,F10.3,A3)') &
     &  ' (A)   Average height (sea bed) :',hmitt(0), &
     &  ' m',hmitv(0),' m   (',hmitw(0),' m)'
       write(iu,'(A33,F11.3,A2,F12.3,A6,F10.3,A3)') &
     &  ' (B)   Average h.   (sea level) :',hmitt(1), &
     &  ' m',hmitv(1),' m   (',hmitw(1),' m)'
       write(iu,'(A33,F11.3,A2,F12.3,A6,F10.3,A3)') &
     &  ' (B*)  Average h. (upper lakes) :',hmitt(2), &
     &  ' m',hmitv(2),' m   (',hmitw(2),' m)'
       write(iu,'(7x,70a1/)') ('-',i=1,70)
       write(iu,'(A24,A27,1p,1e19.9,A5)')'       Covered area ....', &
     &  '. (sea level, integrated) :',Fges,' km^2'
       write(iu,'(A24,A27,1p,1e19.9,A5)')'       Ellipsoid surface', &
     &  ' ........... (analytical) :',Feli*1.d-6,' km^2'
       write(iu,'(A23,A28,1p,1e19.9,A5/)')'       Surface of spher', &
     &  'e ......... (equal volume) :',Fkug*1.d-6,' km^2'
       write(iu,'(A24,A27,1p,1e19.9,A5)')'       Volume correction', &
     &  ' ............. (as per A) :',Vges(0),' km^3'
       write(iu,'(A24,A27,1p,1e19.9,A5)')' (C)   Volume of landmas', &
     &  's ............ (as per B) :',Vges(1),' km^3'
       write(iu,'(A24,A27,1p,1e19.9,A5)')' (C*)  Volume of landmas', &
     &  's + lakes ... (as per B*) :',Vges(2),' km^3'
       write(iu,'(A22,A29,1p,1e14.4,A10/)')'       Volume of upper', &
     &  ' lakes ........... (C* - C) :',Verdl,'      km^3'
       write(iu,'(A24,A27,1p,1e19.9,A5)')' (D)   Earth`s volume ..', &
     &  '.. (ellipsoid, sea level) :',V0,   ' km^3'
       write(iu,'(A23, A28,1p,1e19.9,A5)')' (E)   Earth`s volume +', &
     &  ' landmass ........ (D + C) :',Verdg,' km^3'
       write(iu,'(A24,A27,1p,1e19.9,A5)')' (F)   Earth`s vol. + la', &
     &  'ndm. + lakes ... (D + C*) :',Verds,' km^3'
       write(iu,'(1x,78a1/)') ('=',i=1,78)
      enddo
      go to 200
  100 continue
      do iu=1,6,5
        write(iu,'(/A36,A20)')'  ----->   Insert a correct number:  ',&
     &   'lat-max >= lat-min,'
        write(iu,'(37x,a25//)')'lon-max >= lon-min + 0.05'
      enddo
  200 close(1)
      stop
      end program topo

      subroutine levels(ev0,k,l,ev2)
!-----Erfassung von hoch oder tief liegenden Binnenseen. Falls
!     zutreffend wird die Hoehenangabe 'ev2' korrigiert. (Anmerkung:
!     Die Gebiete duerfen nicht ueber den Nullmeridian reichen, was
!     auch nicht vorkommt. Gegebenenfalls waere eine Loesung, das 
!     Gebiet entlang des Nullmeridians zu teilen.)
      use lakes
      implicit double precision (a-h,o-z)
      ev2 = ev0
      do i=1,nlake
        elev = se(5,i)
        if (il(1,i)<=k.and.k<=il(2,i).and. &
     &      il(3,i)<=l.and.l<=il(4,i)) then
          if (ev0<elev) ev2 = elev; go to 10
        else
          if (i>=nlake.and.ev0<zero) ev2 = zero
        endif
      enddo
   10 return
      end

      subroutine koord(br,x,y)
!-----Input: 'br' geograph. Breite; Output: 'x' Abstand des Brei-
!     tenkreises zur Erdachse, 'y' Abstand des Breitenkreises zur
!     Aequatorebene (mit Vorz.); Berechnung fuer abgeplattete
!     Sphaeroidgestalt.
      use constants
      implicit double precision (a-h,o-z)
      x = a / dsqrt(1.d0 + (c*dtan(br)/a)**2)
      y = c * dsqrt(1.d0 - (x/a)**2)
      if (br<0.d0) y = -y
      return
      end

      subroutine stripe(b,dwi,n,F,Fh)
!-----berechnet die Flaeche F des Streifens (Breite 'dwi') zwischen
!     zwei Breitenkreisen mit den geographischen Breiten b-dwi/2 und
!     b+dwi/2 auf der Ellipsoid-Oberflaeche. Der Winkel dwi ist der
!     geographische Breiten- bzw. Laengenunterschied zweier benach-
!     barter Gitterpunkte im Bogenmass. Je groesser der Parameter 'n'
!     ist, desto genauer ist die numerische Integration der Oberflae-
!     che. (Der Parameter n teilt den Streifen der Flaeche F in 2*n
!     parallele duenne Streifen, wobei n = 100 hier voellig ausrei-
!     chend ist.)
      use constants
      implicit double precision (a-h,o-z)
      ti = 1.d0/dfloat(2*n)
      F  = 0.d0
      call koord(b-0.5d0*dwi,x1,y1)
      do i=1,2*n
        call koord(b+(dfloat(i)*ti-0.5d0)*dwi,x2,y2)
        dbr = dsqrt((x2-x1)**2 + (y2-y1)**2)
        F = F + (x1+x2)*dbr
        if (i.eq.n) Fh = F * pi*dl/dnx
        x1 = x2
        y1 = y2
      enddo
      F = F * pi*dl/dnx
      return
      end
!     Number of lines: 406
