      PROGRAM DATUM_2
!---------------------------------------------------------------------!
! '                                                                 ' !
! '       ===================================================       ' !
! '            CALCULATION OF CALENDAR DATES DUE TO THE             ' !
! '         GREGORIAN AND JULIAN CALENDAR WITH TIME SYSTEMS         ' !
! '        "TERRESTRIAL TIME" (TT) AND "UNIVERSAL TIME" (UT)        ' !
! '       ===================================================       ' !
! '                                                                 ' !
! '        >> Programm "DATUM-2" (Linux-Version, gfortran) <<       ' !
! '              (by H. Jelitto, Hamburg, Jan. 8, 2022)             ' !
! '                                                                 ' !
! '--------------------------- SOME INFO ---------------------------' !
! '                                                                 ' !
! ' The input can be a decimal year, a Julian Ephemeris Day number  ' !
! ' (JDE), or any of the three main dates of the planetary correla- ' !
! ' tion. Note, that for option (1) and (2) (first menu), the same  ' !
! ' decimal year yields slightly different dates if the time period ' !
! ' of the Julian calendar is used, like the year -500. The reason  ' !
! ' is that in option (1) the decimal year input corresponds always ' !
! ' to the Gregorian calendar, whereas in option (2) the decimal    ' !
! ' year depends on the kind of calendar. The values of delta-T and ' !
! ' its error are given to a tenth of a second. This accuracy makes ' !
! ' sense only for the last few centuries. For more distant times   ' !
! ' the numbers have to be rounded accordingly.                     ' !
! '                                                                 ' !
! ' The option "Auto-choice of Jul./Greg. cal. (1)" means that the  ' !
! ' calendar is chosen automatically: The Julian calendar is used   ' !
! ' for the years 4712 BC to 1582 AD and the Gregorian calendar for ' !
! ' all other times. The option "Both calendars (2)" means that for ' !
! ' the same point in time both calendar dates are given.           ' !
! '                                                                 ' !
! ' Originally, the algorithm of Meeus worked only for non-negative ' !
! ' JDE. Here, the algorithm is slightly adapted, so that it is     ' !
! ' valid for both calendars for all times in the past and future.  ' !
! ' However, the allowed points in time are limited to the period   ' !
! ' 1,000,000 BC to 1,000,000 AD.                                   ' !
! '                                                                 ' !
! '-----------------------------------------------------------------' !
! '                                                                 ' !
! ' Copyright (c) 2014-2022 H. Jelitto, with following exceptions:  ' !
! '                                                                 ' !
! ' The calculations are based on a calendar algorithm given in the ' !
! ' book of Jean Meeus: "Astronomical Algorithms", Willmann-Bell,   ' !
! ' Inc., P.O.Box 35025, Richmond, Virginia 23235, USA (page 63).   ' !
! '                                                                 ' !
! ' The subroutine DELTA_T for the calculation of delta-T = TT - UT ' !
! ' accounts for the deceleration of the rotating Earth. The corre- ' !
! ' sponding polynomials up to 7th degree were adapted from the     ' !
! ' "Five Millennium Canon of Solar Eclipses" by Fred Espenak and   ' !
! ' Jean Meeus and released on the "NASA Eclipse Web Site".         ' !
! '                                                                 ' !
! ' USE OF PROGRAM: The program may be used freely for private,     ' !
! ' scientific, and educational purposes. All kinds of commercial   ' !
! ' use without the written permission from the copyright owners/   ' !
! ' authors is not allowed.                                         ' !
! '                                                                 ' !
! '-----------------------------------------------------------------' !
!                                                                     !
!   SOMR MORE INFO:                                                   !
!                                                                     !
!   The results of delta-T = TT - UT are represented by polynomials   !
!   up to 7th degree. They are given on the 'NASA Eclipse Web Site'   !
!   on the page 'Polynomial Expressions for Delta T'. For the calcu-  !
!   lation by F. Espenak and J. Meeus, publications from Morrison,    !
!   Stephenson, and Huber were taken into account. Today, these       !
!   equations seem to yield the exactest values for delta-T as a      !
!   function of time.                                                 !
!                                                                     !
!              The program is used for some calculations              !
!                       in the following books:                       !
!                                                                     !
!                1.    'PYRAMIDEN UND PLANETEN -                      !
!                    Ein vermeintlicher Messfehler                    !
!                      und ein neues Gesamtbild                       !
!                       der Pyramiden von Giza'.                      !
!                                                                     !
!                         ISBN 3-89685-507-7                          !
!                                                                     !
!                2.             Book 2                                !
!                          (in preparation)                           !
!                                                                     !
!                  Wissenschaft und Technik Verlag                    !
!                 Dresdener Str. 26, D-10999 Berlin                   !
!                Tel: 030-6166020, Fax: 030-61660220                  !
!                                                                     !
!                                                                     !
!   More technical and general information is given in the descrip-   !
!   tion "G2-manuals-09-2017.pdf" and also in the Fortran source      !
!   code "DATUM-2.f95".                                               !
!                                                                     !
!---------------------------------------------------------------------!

      IMPLICIT DOUBLE PRECISION (A-H,O-Z)
      DIMENSION IDA(7),DA(7),ZJDP(3)
      CHARACTER(1) :: B1,B2,BZ
      CHARACTER(4) :: DMO
      CHARACTER(10) :: wd
      CHARACTER(21) :: DUMMY,DD,D(0:13)
      CHARACTER(30) :: T0,T1,T2
      data ZJDP/2849035.77863d0,2849067.30624d0,2849079.76330d0/
      data D/'  INCORRECT INPUT -->','  dec. year (Greg.) :', &
             '  dec. year (Jul.)  :','               JDE  :', &
             '  date (Jul., TT)   :','  date (Jul., UT)   :', &
             '  date (Greg., TT)  :','  date (Greg., UT)  :', &
             '  mean trend (UT)   :','  delta-T = TT-UT   :', &
             '  error of delta-T  :','  decimal year      :', &
             '  length of day     :','    (day of the week:'/
!-----HAUPTPROGRAMM
      open(unit=1,file='out.txt')

! - - Programm-Eingabe
      ICC = 1; ir = 1; BZ = ')'
      T1 = ' Julian epoch (fixed to J2000)'
      T2 = '            program P3, book 1'
      WRITE(6,'(//,4X,''-->   Auto-choice of Jul./Greg. '', &
        & ''cal. (default)  (1)'')') 
      WRITE(6,'(4X,''-->   Both calendars'',28X,''(2)'')') 
      WRITE(6,'(4X,''-->   Info'',38X,''(3)  : '')' , &
        & advance='no') 
      READ(*,'(A21)') DUMMY
      imod = 1
      IF (DUMMY=='2') imod = 2
      IF (DUMMY=='3') imod = 3
      IF (imod==3) then
        call info; go to 100
      ENDIF
      WRITE(6,'(/,4X,''-->   General input    ---  '', &
        & ''year     (default)  (1)'')')
      WRITE(6,'(4X,''-->     "       "      ---  JDE'',17X,''(2)'')')
      WRITE(6,'(4X,''-->   Predefined date  ---  date of chambers'', &
        & 4X,''(3)'')')
      WRITE(6,'(4X,''-->       "       "    ---'', &
        & ''  middle of transit   (4)'')')
      WRITE(6,'(4X,''-->       "       "    ---'', &
        & ''  date of pyramids    (5)  : '')' &
        ,advance='no')
      READ(*,'(A21)') DUMMY
      iopt = 1
      IF (DUMMY=='2') iopt = 2
      IF (DUMMY=='3') iopt = 3
      IF (DUMMY=='4') iopt = 4
      IF (DUMMY=='5') iopt = 5
      open(unit=1,file='out.txt')
      do iu=1,6,5
        write(iu,'(//1X,78(''=''))')
        IF (iopt<=2) write(iu,'(6X,''Program "DATUM-2" - '', &
          & ''computation of calendar dates (used in P4 and P5)'')')
        IF (iopt==3) write(iu,'(3X,''"DATUM-2"  -  Constellation'', &
          & '' 12: date of chamber alignment in Great Pyramid'')')
        IF (iopt==4) write(iu,'(6X,''"DATUM-2"  -  Constellation'', &
          & '' 12: minimum separation of Mercury transit'')')
        IF (iopt==5) write(iu,'(8X,''"DATUM-2"  -  Constellation'', &
          & '' 12: date of pyramid positions in Giza'')')
        write(iu,'(1X,78(''=''))')
      enddo 
      IF (iopt==2) THEN
        WRITE(6,'(A21,''      '')',advance='no') D(3)
        READ(*,*) ZJD
        IF (ZJD>=0.d0.AND.ZJD<2299160.5d0) ir = 2
        WRITE(1,'(A21,F19.5)') D(3),ZJD
      ELSEIF (iopt>=3.AND.iopt<=5) THEN
        ZJD = ZJDP(iopt-2)
        do iu=1,6,5
          write(iu,'(A21,F19.5)') D(3),ZJD
        enddo
      ELSE
        WRITE(6,'(A21,''      '')',advance='no') D(21-10*imod)
        READ(*,*) YEAR
        IF (YEAR>=-4712.d0.AND.YEAR<1582.7854097d0) ir = 2
        WRITE(1,'(A21,F19.5)') D(1),YEAR
      ENDIF

      do 99 iu=1,6,5
! - - Ueberpruefen der Zeiteingabe
      IF ((iopt==1.AND.(YEAR<-1.d6.OR.YEAR>1.d6)).OR.(iopt==2 &
        .AND.(ZJD<-363521440.d0.OR.ZJD>366963560.d0))) THEN
        write(iu,'(/A21, &
         & 4X,'' -1 000 000   <=  year  <=    1 000 000    and''/ &
         & 24X,''-363 521 440.  <=   JDE  <=  366 963 560.''//)')D(0)
        GO TO 99
      ENDIF

! - - Berechnung von Julian Ephemeris Day bzw. dezimaler Jahreszahl
      IF (imod==1.and.iopt==1) THEN
        call cephim(1,ir,ZJD,YEAR,WD)
        WRITE(iu,'(A21,F19.5,6x,A21,A10,A1)') D(3),ZJD,D(13),WD,BZ
      ELSEIF (imod==1.and.iopt>=2) THEN
        call cephim(2,ir,ZJD,YEAR,WD)
        WRITE(iu,'(A21,F19.5,6x,A21,A10,A1)') D(ir),YEAR,D(13),WD,BZ
      ELSEIF (imod==2.and.iopt==1) THEN
        CALL cephim(1,1,ZJD,YEAR,WD) ! (TT, Greg., JDE)
        CALL cephim(2,2,ZJD,YEARJ,WD) ! (TT, Julian., dec. year)
        WRITE(iu,'(A21,F19.5)') D(2),YEARJ
        WRITE(iu,'(A21,F19.5,6x,A21,A10,A1)') D(3),ZJD,D(13),WD,BZ
      ELSEIF (imod==2.and.iopt>=2) THEN
        CALL cephim(2,1,ZJD,YEAR,WD) ! (TT, Greg., dec. year)
        WRITE(iu,'(A21,F19.5)') D(1),YEAR
        CALL cephim(2,2,ZJD,YEARJ,WD) ! (TT, Julian., dec. year)
        WRITE(iu,'(A21,F19.5,6x,A21,A10,A1)') D(2),YEARJ,D(13),WD,BZ
      ENDIF

! - - Datumsberechnung (TT und UT)
      write(iu,'(1X,78(''-''))')
      YY = YEAR
      IF (IMOD==1.AND.IR==2) call cephim(2,1,ZJD,YY,WD)
      CALL DELTA_T(1,YY,del1)
      CALL DELTA_T(2,YY,del2)
      MEAN = 0; IF (YEAR>=-500.d0.and.YEAR<=2150.d0) MEAN = 1
      i1 = 1; i2 = 6
      if (imod==1.and.ir==1) i2 = 3
      if (imod==1.and.ir==2) i1 = 4
      DO II=i1,i2
        IF (II==1) THEN
          CALL JDEDATUM(ZJD,1,IDA,DA,DMO) ! (Greg., TT)
          DD = D(6); B1 = ' '; B2 = ' '
        ELSEIF (II==2) THEN
          CALL JDEDATUM(ZJD-del1/86400.d0,1,IDA,DA,DMO) ! (Greg., UT)
          DD = D(7) 
        ELSEIF (II==3.AND.MEAN==1) THEN
          CALL JDEDATUM(ZJD-del2/86400.d0,1,IDA,DA,DMO) ! (mean trend)
          DD = D(8); B1 = '('; B2 = ')'
        ELSEIF (II==4) THEN
          CALL JDEDATUM(ZJD,2,IDA,DA,DMO) ! (Julian., TT)
          DD = D(4); B1 = ' '; B2 = ' '
        ELSEIF (II==5) THEN
          CALL JDEDATUM(ZJD-del1/86400.d0,2,IDA,DA,DMO) ! (Julian., UT)
          DD = D(5)
        ELSEIF (II==6.AND.MEAN==1.AND.IMOD==1.AND.IR==2) THEN
          CALL JDEDATUM(ZJD-del2/86400.d0,2,IDA,DA,DMO) ! (mean trend)
          DD = D(8); B1 = '('; B2 = ')'
        ENDIF
        IF ((II/=3.AND.II/=6).OR.(II==3.AND.MEAN==1).OR. &
             (II==6.AND.MEAN==1.AND.IMOD==1.AND.IR==2)) THEN
          IF (IDA(3)>-99995.d0.AND.IDA(3)<=99995.d0) THEN
            write(iu,'(A21,4X,A1,F3.0,1X,A4,I6,'','',I6,'':'', &
              & I2,'':'',I2,A1,4X,''(dec. day:'',F9.5,'')'')') &
              & DD,B1,DA(7),DMO,(IDA(I),I=3,6),B2,DA(1)
          ELSE
            write(iu,'(A21,3X,A1,F3.0,1X,A4,I8,'','',I5,'':'', &
              & I2,'':'',I2,A1,4X,''(dec. day:'',F9.5,'')'')') &
              & DD,B1,DA(7),DMO,(IDA(I),I=3,6),B2,DA(1)
          ENDIF
        ENDIF
      ENDDO
      write(iu,'(1X,78(''-''))')

! - - Ausgabe von delta-T = TT - UT und Fehlerabschaetzung
      CALL convert(del1,iday,ihour,imin,sec)
      write(iu,'(A21,F15.1,'' sec ='',I6,'' days'',I3,'' hrs'', &
        & I3,'' min'',F5.1,'' sec'')') D(9),del1,iday,ihour,imin,sec
      CALL sigma_T(YEAR,ddl)
      CALL convert(ddl,iday,ihour,imin,sec)
      write(iu,'(A21,F15.1,'' sec ='',I6,'' days'',I3,'' hrs'', &
        & I3,'' min'',F5.1,'' sec'')') D(10),ddl,iday,ihour,imin,sec
      ds = 86400.d0 + 0.64d0*(year-1820.d0)/36524.25d0
      CALL convert(ds,iday,ihour,imin,sec)
      write(iu,'(A21,F15.3,'' sec ='',I6,'' days'',I3,'' hrs'', &
        & I3,'' min'',F7.3,'' sec'')')D(12),ds,iday,ihour,imin,sec
      write(iu,'(1X,78(''=''))')

! - - Alternative Gleichungen (dez. Jahr --> JDE --> Datum)
      write(iu,'(/25X,''Other functions "year(JDE)"''/ &
            & 1X,78(''-''))')
      Do II=1,2
!       Julianische Epoche (hauptsaechlich 1900 - 2100)
        IF (II==1) THEN
          CALL cephim(2,3,ZJD,YEARS,WD)
          T0 = T1
        ELSE
!       Buch 1, Programm P3
          CALL cephim(2,4,ZJD,YEARS,WD)
          T0 = T2
        ENDIF
        write(iu,'(A21,F19.5,8X,A30)') D(11),YEARS,T0
      ENDDO
      write(iu,'(1X,78(''-'')//)')

! - - Programm-Ende
      close(iu)
   99 enddo
  100 STOP
      END PROGRAM DATUM_2

      SUBROUTINE JDEDATUM(ZJD,ICC,IDA,DA,DMO)
!-----BERECHNUNG VON DATUM UND UHRZEIT---------------------------------
!     INDICES  1: dez.Tag, 2: Monat,   3: Jahr, 4: Stunde,
!              5: Minute,  6: Sekunde, 7: Tag
!     ICC: Gregorian Kalender (1), Julian Kalender (2) 
      IMPLICIT DOUBLE PRECISION (A-H,O-Z)
      DIMENSION IDA(7),DA(7)
      CHARACTER(4) :: MONAT(12),DMO
      DATA MONAT/'Jan.','Feb.','Mar.','Apr.','May ','June', &
                 'July','Aug.','Sep.','Oct.','Nov.','Dec.'/
      Z = SDINT(ZJD + 0.5D0)
      F = ZJD + 0.5D0 - Z
      IF (ICC==2) THEN
         A = Z
      ELSE
         ALPHA = SDINT((Z -  1867216.25D0)/36524.25D0)
         A = Z + 1.D0 + ALPHA - SDINT(ALPHA/4.D0)
      ENDIF
      B = A + 1524.D0
      C = SDINT((B - 122.1D0)/365.25D0)
      D = SDINT(365.25D0 * C)
      E = SDINT((B - D)/30.6001D0)
      DA(1) = B - D - SDINT(30.6001D0*E) + F + 5.d-9
      IF (E<14.D0) THEN
         DA(2) = E - 1.D0
      ELSE
         IF (E==14.D0.OR.E==15.D0) THEN
            DA(2) = E - 13.D0
         ELSE
            DA(2) = 999.D0
         ENDIF
      ENDIF
      M = IDNINT(DA(2))
      IF (M>2) THEN
         DA(3) = C - 4716.D0
      ELSE
         IF (M==1.OR.M==2) THEN
            DA(3) = C - 4715.D0
         ELSE
            DA(3) = 9999999999999.D0
         ENDIF
      ENDIF
      ST  = DA(1) - SDINT(DA(1))
      DST = ST*24.D0
      DA(4) = SDINT(DST)
      DA(5) = (DST - SDINT(DST))*60.D0
      DA(6) = (DA(5) - SDINT(DA(5)))*60.D0
      DA(7) = SDINT(DA(1))                  ! day
      IDA(3) = IDNINT(DA(3))                ! year
      IDA(4) = IDNINT(DA(4))                ! hours
      IDA(5) = IDNINT(DA(5)-0.5d0+1.d-10)   ! minutes
      IDA(6) = IDNINT(DA(6))                ! seconds
      IMO    = IDNINT(DA(2))                ! month

!     Korrektur der Darstellung
!     (Beispiel: Uhrzeit 13:44:60 wird zu 13:45:00)
      do i=6,5,-1
        if (ida(i)>=60) then
          ida(i) = ida(i) - 60
          ida(i-1) = ida(i-1) + 1
        endif
      enddo
      if (ida(4)>=24) then
        ida(4) = ida(4) - 24
        da(1) = da(1) + 1.d0
        da(7) = sdint(da(1))
      endif
      if ((dabs(da(7)-32.d0)<=1.d-8.and.(imo==1.or.imo==3 &
         .or.imo==5.or.imo==7.or.imo==8.or.imo==10.or.imo==12)).or. &
          (dabs(da(7)-31.d0)<=1.d-8.and.(imo==4.or.imo==6.or.imo==9 &
         .or.imo==11)).or.(dabs(da(7)-30.d0)<=1.d-8.and.imo==2)) then
        do k=30,32
          q=dfloat(k); if (dabs(da(7)-q)<=1.d-8) da(1) = da(1)+1.d0-q
        enddo
        da(7) = sdint(da(1)); imo = imo + 1
        if (imo==13) then
          imo = 1; da(3) = da(3) + 1.d0
          ida(3) = idnint(da(3))
        endif
      endif
      dmo = monat(imo)
      END SUBROUTINE

      double precision function SDINT(x)
!-----Step function, replacing some integer-functions in the sub-
!     routine "JDEDATUM" in order to expand the domain of definition
      real(8) :: x
      sdint = dint(x)
      if (x<0.d0.and.dmod(x,1.d0)/=0.d0) sdint = sdint - 1
      end function

      SUBROUTINE DELTA_T(im,y,del)
!-----Berechnung von delta-T zur Umwandlung TT -> UT-------------------
!     im = 1: Berechnung mit Polynomen bis zu 7. Grades nach
!             Fred Espenak und Jean Meeus (NASA Eclipse Web Site,
!             Polynomial Expressions for Delta T, 2004)
!     im = 2: 'Langzeit'-Berechnung bzw. gemittelter Wert, die-
!             selben Autoren (nur quadratisches Polynom, siehe
!             unten)
      IMPLICIT DOUBLE PRECISION (A-H,O-Z)
      if (y>-500.d0.and.y<=500.d0.and.im==1) then
         u = y/100.d0
         del = 10583.6d0 - 1014.41d0 * u + 33.78311d0  * u**2 &
               - 5.952053d0 * u**3 - 0.1798452d0 * u**4 &
               + 0.022174192d0 * u**5 + 0.0090316521d0 * u**6
      elseif (y>500.d0.and.y<=1600.d0.and.im==1) then
         u = (y-1000.d0)/100.d0
         del = 1574.2d0 - 556.01d0 * u + 71.23472d0 * u**2 &
               + 0.319781d0 * u**3 - 0.8503463d0 * u**4 &
               - 0.005050998d0 * u**5 + 0.0083572073d0 * u**6
      elseif (y>1600.d0.and.y<=1700.d0.and.im==1) then
         t = y - 1600.d0
         del = 120.d0 - 0.9808d0 * t - 0.01532d0 * t**2 &
               + t**3 / 7129.d0
      elseif (y>1700.d0.and.y<=1800.d0.and.im==1) then
         t = y - 1700.d0
         del = 8.83d0 + 0.1603d0 * t - 0.0059285d0 * t**2 &
               + 0.00013336d0 * t**3 - t**4 / 1174000.d0
      elseif (y>1800.d0.and.y<=1860.d0.and.im==1) then
         t = y - 1800.d0
         del = 13.72d0 - 0.332447d0 * t + 0.0068612d0 * t**2 &
               + 0.0041116d0 * t**3 - 0.00037436d0 * t**4 &
               + 0.0000121272d0 * t**5 - 0.0000001699d0 * t**6 &
               + 0.000000000875d0 * t**7
      elseif (y>1860.d0.and.y<=1900.d0.and.im==1) then
         t = y - 1860.d0
         del = 7.62d0 + 0.5737d0 * t - 0.251754d0 * t**2 &
               + 0.01680668d0 * t**3 - 0.0004473624d0 * t**4 &
               + t**5 / 233174.d0
      elseif (y>1900.d0.and.y<=1920.d0.and.im==1) then
         t = y - 1900.d0
         del = -2.79d0 + 1.494119d0 * t - 0.0598939d0 * t**2 &
               + 0.0061966d0 * t**3 - 0.000197d0 * t**4
      elseif (y>1920.d0.and.y<=1941.d0.and.im==1) then
         t = y - 1920.d0
         del = 21.20d0 + 0.84493d0 * t - 0.076100d0 * t**2 &
               + 0.0020936d0 * t**3
      elseif (y>1941.d0.and.y<=1961.d0.and.im==1) then
         t = y - 1950.d0
         del = 29.07d0 + 0.407d0 * t - t**2/233.d0 + t**3/2547.d0
      elseif (y>1961.d0.and.y<=1986.d0.and.im==1) then
         t = y - 1975.d0
         del = 45.45d0 + 1.067d0 * t - t**2/260.d0 - t**3/718.d0
      elseif (y>1986.d0.and.y<=2005.d0.and.im==1) then
         t = y - 2000.d0
         del = 63.86d0 + 0.3345d0 * t - 0.060374d0 * t**2 &
               + 0.0017275d0 * t**3 + 0.000651814d0 * t**4 &
               + 0.00002373599d0 * t**5
      elseif (y>2005.d0.and.y<=2050.d0.and.im==1) then
         t = y - 2000.d0
         del = 62.92d0 + 0.32217d0 * t + 0.005589d0 * t**2
      elseif (y>2050.d0.and.y<=2150.d0.and.im==1) then
         del = -20.d0 + 32.d0 * ((y-1820.d0)/100.d0)**2 &
               - 0.5628d0 * (2150.d0 - y)
      else
! . . .  gemittelter Wert, zur Extrapolation in die
!        ferne Vergangenheit und die ferne Zukunft
         u = (y - 1820.d0)/100.d0
         del = -20.d0 + 32.d0 * u**2
      endif

! . . Spaetere Korrektur (Siehe dazu die NASA Eclipse Web Site.)
      if (y<1955.d0 .or.y>2005.d0) del = del-1.2932d-5*(y-1955.d0)**2

! . . alternative 'Langzeit'-Delta-T-Berechnung
!     (Meeus, 1991, after Morrison & Stephenson, 1982)
!c    del = -15.d0 + (zjd - 2382148.d0)**2 / 41048480.d0
!     ('zjd' muss in Parameterliste von DELTA_T eingefuegt werden!)
      end subroutine

      SUBROUTINE sigma_T(YEAR,sigma)
!-----Fehlerabschaetzung fuer delta-T (alle Zeiten)--------------------
!     gemaess NASA Eclipse Web Site - Historical Values of delta-T
!     (Tab.1). Die Standardfehler wurden abschnittsweise mit linea-
!     ren und quadratischen Funktionen gefittet.
!     Erste und letzte Funktion fuer sigma nach Huber, P. J.: Modeling
!     the Length of Day and ..., Astronomical Amusements, Rome (2000)
!     Zweite Funktion fuer sigma nach Morrison und Stephenson: Histo-
!     rical Values ..., J. Hist. Astron. 35/3, No. 120 (2004) 327–336
      IMPLICIT DOUBLE PRECISION (A-H,O-Z)
      IF (YEAR<-1000.d0) THEN
        dJ = DABS(YEAR + 500.d0)
        sigma = 0.36525d0 * dJ * DSQRT((dJ*0.058/3.d0) &
                 * (1.d0 + dJ/2500.d0))
      ELSEIF (YEAR>=-1000.d0.AND.YEAR<1300.d0) THEN
        sigma = 0.00008d0 * (YEAR-1820.d0)**2
      ELSEIF (YEAR>=1300.d0.AND.YEAR<1600.d0) THEN
        sigma = 20.d0
      ELSEIF (YEAR>=1600.d0.AND.YEAR<1700.d0) THEN
        sigma = -0.15d0 * YEAR + 260.d0
      ELSEIF (YEAR>=1700.d0.AND.YEAR<1750.d0) THEN
        sigma = -0.06d0 * YEAR + 107.d0
      ELSEIF (YEAR>=1750.d0.AND.YEAR<1800.d0) THEN
        sigma = -0.02d0 * YEAR + 37.d0
      ELSEIF (YEAR>=1800.d0.AND.YEAR<1900.d0) THEN
        sigma = 0.00009d0 * (YEAR-1900.d0)**2 + 0.1d0
      ELSEIF (YEAR>=1900.d0.AND.YEAR<2012.d0) THEN
        sigma = 0.1d0
      ELSE
        dJ = DABS(YEAR - 2012.d0)
        sigma = 0.36525d0 * dJ * DSQRT((dJ*0.058/3.d0) &
                 * (1.d0 + dJ/2500.d0)) + 0.1d0
      ENDIF
      end subroutine

      SUBROUTINE cephim(IOPT,ICC,ZJD,YEAR,wd)
!-----Berechnung des dezimalen Jahres----------------------------------
!     IOPT = 1:   dezimales Jahr --> JDE
!      "  >= 2:   JDE --> dezimales Jahr
!     ICC  = 1:   Gregorianischer Kalender
!      "   = 2:   Julianischer Kalender
!      "   = 3:   Julianische Epoche (korreliert mit J2000)
!      "   = 4:   Gleichung aus Buch 1 und Programm P3
      IMPLICIT DOUBLE PRECISION (A-H,O-Z)
      character(10) :: wd
      DATA A/0.D0/,B/0.D0/,C/0.D0/  ! (vorab initialisiert
!       zur Vermeidung von Warnmeldungen beim Parameter "-Wall")
      IF (ICC==1) THEN
!       gemaess Gregorianischem Kalender (Programm P4)
        A = 365.2425d0
        B = 2451545.d0
        C = 2000.d0
      ELSEIF (ICC==2) THEN
!       gemaess Julianischem Kalender (Programm P4)
        A = 365.25d0 
        B = 0.d0
        C = -4712.d0
      ELSEIF (ICC==3) THEN
!       Julianische Epoche (angeglichen an J2000.0)
        A = 365.25d0
        B = 2451545.d0
        C = 2000.d0
      ELSEIF (ICC==4) THEN
!       Gleichung aus Buch 1 ("Pyramiden und Planeten", Programm P3)
        A = 365.248d0
        B = 0.d0
        C = -4711.9986d0
      ENDIF
      IF (IOPT==1) ZJD = A * (YEAR - C) + B
      IF (IOPT>=2) YEAR = (ZJD - B)/A + C
      call weekday(ZJD,wd)
      end subroutine

      subroutine weekday(ZJD,wd)
!-----Berechnung des Wochentages
      implicit double precision(a-h,o-z)
      character(10) :: wday(0:6),wd
      data wday/'    Sunday','    Monday','   Tuesday',' Wednesday', &
                '  Thursday','    Friday','  Saturday'/
      wd = wday(idnint(dmod(dint(ZJD + 700000001.5d0),7.d0)))
      end subroutine

      SUBROUTINE convert(del,iday,ihour,imin,sec)
!-----Umwandlung eines Zeitraums---------------------------------------
!     Sekunden  -->  Tage, Stunden, Minuten und Sekunden
      IMPLICIT DOUBLE PRECISION (A-H,O-Z)
      zday  = dint(del/86400.d0)
      iday  = idnint(zday)
      dl    = del - zday * 86400.d0
      zhour = dint(dl/3600.d0)
      ihour = idnint(zhour)
      dl    = dl - zhour * 3600.d0
      zmin  = dint(dl/60.d0)
      imin  = idnint(zmin)
      sec   = dl - zmin * 60.d0
      end subroutine

      subroutine info
!-----Information zu den Copyrights und zum Programm-------------------
      character(1)  :: dummy
      character(65) :: itext(55)
      open(unit=10,file='DATUM-2.f95')
      do i=1,3; read(10,*); enddo
      do i=1,55; read(10,*) dummy, itext(i); enddo
      close(10)
      write(*,*)
      do ia=1,6,5
      write(ia,*) 
        do i=1,55; write(ia,'(7x,a65)') itext(i); enddo
        write(ia,*)
      enddo
      end subroutine
!     Number of lines: 567
